//
//  OPGameController.m
//  The Obfuscated Path
//
//  Created by ss2cire on 12/6/08.
//  Copyright 2008 GryphonClaw Software. All rights reserved.
//

#import "OPGameController.h"
#import "OPRoom.h"
#import "OPRoomLayer.h"
#import "OPTextureManager.h"
#import "OPSprite.h"
#import "OPPlayer.h"
#import "OPTypes.h"
#import "OPApplication.h"
#import "OPMap.h"
#import "OPRoom.h"
#import "OPHUDView.h"
#import "OPTrigger.h"
#import "OPItem.h"
#import "OPEnemy.h"

@implementation OPGameController
- (id)init
{
	self = [super init];
	if(self != nil)
	{
		[self loadTextureList];
		scrollingRoom = NO;
		HUDView = [[OPHUDView alloc] init];
		player = [[OPPlayer alloc] initWithLocation:OPMakePoint(32, 64) andSize:OPMakeSize(32, 32)];
		[self loadPlayerTextures];
		[self loadEnemyTextures];

		eventStateDictionary = [[NSMutableDictionary alloc] init];
		[self loadGameMaps];
		[[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(spawnEnemy:) name:OPSpawnEnemyNotification object:nil];
		[[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(createItem:) name:OPCreateItemNotification object:nil];
	}
	return self;
}

- (void)loadGameMaps
{
	if(gameMaps != nil) {
		[gameMaps release];
		gameMaps = nil;
	}
	gameMaps = [[NSMutableDictionary alloc] init];
	
	NSDictionary	*mapsPlist = [NSDictionary dictionaryWithContentsOfFile:[[NSBundle mainBundle] pathForResource:@"Maps" ofType:@"plist"]];
	NSEnumerator	*enumerator = [[mapsPlist objectForKey:@"Maps"] objectEnumerator];
	OPMap			*aMap = nil;
	NSString		*aMapName = nil;
	while((aMapName = [enumerator nextObject]) != nil) {
		aMap = [[[OPMap alloc] initWithMap:aMapName] autorelease];
		[gameMaps setObject:aMap forKey:aMapName];
	}
	currentMap = [gameMaps objectForKey:[mapsPlist objectForKey:@"FirstMap"]];
	[currentMap setCurrentRoomAtX:0 andY:0];
}

- (void)dealloc
{
	[player release];
	currentMap = nil;
	[gameMaps release];
	[HUDView release];
	[eventStateDictionary release];
	[super dealloc];
}

- (void)draw
{
	[currentMap draw];
	[HUDView draw];
	[player drawStatus];
	[player drawWeapon];
}

- (void)update
{
	if(!scrollingRoom) {
		[player update];
		[self checkPlayerCollisions];
		[self checkEnemyCollisions];
		if([player isAttacking]) {
			[self checkWeaponCollision];
		}
		[self checkEvents];
	} else {
		[currentMap scroll];
		scrollingRoom = [currentMap scrolling];
	}

	[currentMap update];
}

- (void)checkPlayerCollisions
{
	OPSprite *spriteToCheck = nil;

	spriteToCheck = [currentMap tileAtX:[player locationX] andY:[player locationY] ] ;//get the sprite directly to the left
	[player checkCollision:spriteToCheck];

	spriteToCheck = [currentMap tileAtX:[player locationX] + 24 andY:[player locationY] ] ;//get sprite to the right of the player (or close)
	[player checkCollision:spriteToCheck];

	spriteToCheck = [currentMap tileAtX:[player locationX] andY:[player locationY] + 24 ] ;//get the sprite above the player
	[player checkCollision:spriteToCheck];

	spriteToCheck = [currentMap tileAtX:[player locationX] + 24 andY:[player locationY] + 24 ] ;//get the sprite above and to the right
	[player checkCollision:spriteToCheck];

	[self checkRoomEdgeCollision];
	[self checkItemCollision];
}

- (void)checkEnemyCollisions
{
	OPSprite *spriteToCheck = nil;
	OPEnemy *currentEnemy = nil;
	
	NSEnumerator	*enumerator = [[[currentMap currentRoom] enemies] objectEnumerator];
	while((currentEnemy = [enumerator nextObject]) != nil) {
		spriteToCheck = [currentMap tileAtX:[currentEnemy locationX] andY:[currentEnemy locationY] ] ;//get the sprite directly to the left
		if([currentEnemy checkCollision:spriteToCheck]) {
			[currentEnemy setMovingLeft:NO];
			[currentEnemy setNewDirectionExcluding:OPRight];
		}
		
		spriteToCheck = [currentMap tileAtX:[currentEnemy locationX] + 24 andY:[currentEnemy locationY] ] ;//get sprite to the right (or close)
		if([currentEnemy checkCollision:spriteToCheck]) {
			[currentEnemy setMovingRight:NO];
			[currentEnemy setNewDirectionExcluding:OPLeft];
		}
		
	
		spriteToCheck = [currentMap tileAtX:[currentEnemy locationX] andY:[currentEnemy locationY] + 24 ] ;//get the sprite above
		if([currentEnemy checkCollision:spriteToCheck]) {
			[currentEnemy setMovingUp:NO];
			[currentEnemy setNewDirectionExcluding:OPDown];
		}
		
		spriteToCheck = [currentMap tileAtX:[player locationX] + 24 andY:[currentEnemy locationY] + 24 ] ;//get the sprite above and to the right
		if([currentEnemy checkCollision:spriteToCheck]) {
			[currentEnemy setMovingDown:NO];
			[currentEnemy setNewDirectionExcluding:OPUp];
		}
	}

	enumerator = [[[currentMap currentRoom] enemies] objectEnumerator];
	while((currentEnemy = [enumerator nextObject]) != nil) {
		if(NSIntersectsRect([currentEnemy collisionRect], [player collisionRect])) {
			[player hurt];
		}
	}
}

- (void)checkWeaponCollision
{
	if([[player weapon] isVisible]) {
		OPRect swordRect = [[player weapon] collisionRect];
		OPEnemy *currentEnemy = nil;
		NSMutableArray *enemiesToRemove = [[NSMutableArray alloc] init];
		NSEnumerator	*enumerator = [[[currentMap currentRoom] enemies] objectEnumerator];
		while((currentEnemy = [enumerator nextObject]) != nil) {
			if(NSIntersectsRect([currentEnemy collisionRect], swordRect)) {
				if([currentEnemy hurt]) {
					[enemiesToRemove addObject:currentEnemy];
				}
			}
		}
		[[[currentMap currentRoom] enemies] removeObjectsInArray:enemiesToRemove];
	}
}

- (void)checkEvents
{
	OPRoom *currentRoom = [currentMap currentRoom];
	NSArray *roomTriggers = [currentRoom triggers];

	NSEnumerator *eventEnumerator = [roomTriggers objectEnumerator];
	OPTrigger *aTrigger = nil;
	while((aTrigger = [eventEnumerator nextObject]) != nil) {
		if(![aTrigger isDone]) {
			BOOL intersectedBefore = ([eventStateDictionary objectForKey:[aTrigger identifier]] != nil);
			BOOL intersectsNow     = NSIntersectsRect([player collisionRect], [aTrigger collisionRect]);

			 if(!intersectedBefore && intersectsNow) { //entered trigger {
				 [self runEvent:aTrigger];
				 [eventStateDictionary setObject:@"" forKey:[aTrigger identifier]];
			} else if(intersectedBefore && !intersectsNow) { //exited trigger
				[eventStateDictionary removeObjectForKey:[aTrigger identifier]];
			}
		}
	}

	roomTriggers = nil;
	currentRoom = nil;
}

- (void)runEvent:(OPTrigger *)trigger
{
	if(![trigger repeatable]) {
		[trigger setDone:YES];
	}
	NSArray *actions = [trigger actions];
	NSEnumerator *actionsEnumerator = [actions objectEnumerator];
	id anAction = nil;
	while((anAction = [actionsEnumerator nextObject]) != nil) {
		NSString *actionString = [anAction objectForKey:@"action"];
		NSArray *arguments = [anAction objectForKey:@"args"];
		if([actionString isEqualToString:@"loadMap"]) {
			[self loadMap:[arguments objectAtIndex:0]];
			[eventStateDictionary setObject:@"" forKey:[arguments objectAtIndex:0]];
			[eventStateDictionary removeObjectForKey:[trigger identifier]];
		} else if([actionString isEqualToString:@"setRoom"]) {
			[self setRoom:OPMakePoint([[arguments objectAtIndex:0] intValue], [[arguments objectAtIndex:1] intValue])];
		} else if([actionString isEqualToString:@"movePlayer"]) {
			[self movePlayer:OPMakePoint([[arguments objectAtIndex:0] intValue], [[arguments objectAtIndex:1] intValue])];
			[eventStateDictionary removeObjectForKey:[trigger identifier]];
		} else if([actionString isEqualToString:@"spawnItem"]) {
			[self spawnItem:trigger];
		} else if([actionString isEqualToString:@"removeTree"]) {
			[self removeTree:trigger];
		} else if([actionString isEqualToString:@"hurtPlayer"]) {
			[self hurtPlayer:trigger];
		}
	}
}

- (void)checkRoomEdgeCollision
{
	OPRect rect = [player collisionRect];
	OPRect roomRect = OPMakeRect(0.0, 0.0, 32 * 16, 32 * 10.5);
	int roomTop = (roomRect.origin.y + roomRect.size.height);
	int roomLeft = 0;
	int roomRight = (roomRect.origin.x + roomRect.size.width);
	int roomBottom = 0;
	if([player isMovingUp]) {
		if((rect.origin.y + rect.size.height) > roomTop) {
			scrollingRoom = YES;
			[currentMap setScrolling:scrollingRoom direction:OPUp];
		}
	} else if([player isMovingDown]) {
		if(rect.origin.y < roomBottom) {
			scrollingRoom = YES;
			[currentMap setScrolling:scrollingRoom direction:OPDown];
		}
	}
	if([player isMovingLeft]) {
		if(rect.origin.x < roomLeft) {
			scrollingRoom = YES;
			[currentMap setScrolling:scrollingRoom direction:OPLeft];
		}
	} else if([player isMovingRight]) {
		if((rect.origin.x + rect.size.width) > roomRight) {
			scrollingRoom = YES;
			[currentMap setScrolling:scrollingRoom direction:OPRight];
		}
	}
}

- (void)checkItemCollision
{
	NSMutableArray *theItems = [[currentMap currentRoom] items];
	NSEnumerator *enumerator = [theItems objectEnumerator];
	id anItem = nil;
	BOOL itemFound = NO;
	unsigned int itemToRemoveIndex = 0;
	while((anItem = [enumerator nextObject]) != nil) {
		if(NSIntersectsRect([player collisionRect], [anItem collisionRect])) {
			itemFound = YES;
			break;
		}
		itemToRemoveIndex++;
	}
	if(itemFound) {
		[self doActionForItem:[theItems objectAtIndex:itemToRemoveIndex]];
		[theItems removeObjectAtIndex:itemToRemoveIndex];
	}
}

- (void)doActionForItem:(OPItem *)anItem
{
	NSString *itemType = [anItem type];
	if([itemType isEqualToString:@"heart"]) {
		[player heal];
		[player heal];
	}
}

- (void)keyDown:(NSEvent *)event
{
	NSString *characters;
	characters = [event characters];
	
	unichar character;
	character = [characters characterAtIndex: 0];
	
	switch(character)
	{
		case NSUpArrowFunctionKey:
			[player setMovingUp:YES];
			break;
		case NSDownArrowFunctionKey:
			[player setMovingDown:YES];
			break;
		case NSLeftArrowFunctionKey:
			[player setMovingLeft:YES];
			break;
		case NSRightArrowFunctionKey:
			[player setMovingRight:YES];
			break;
		default:
			break;
	}
}

- (void)keyUp:(NSEvent *)event
{
	NSString *characters;
	characters = [event characters];

	unichar character;
	character = [characters characterAtIndex: 0];

	switch(character)
	{
		case NSUpArrowFunctionKey:
			[player setMovingUp:NO];
			break;
		case NSDownArrowFunctionKey:
			[player setMovingDown:NO];
			break;
		case NSLeftArrowFunctionKey:
			[player setMovingLeft:NO];
			break;
		case NSRightArrowFunctionKey:
			[player setMovingRight:NO];
			break;
		case ' ':
			[player attack];
			break;
		default:
			break;
	}
}

- (void)mouseDown:(NSEvent *)event
{
}

- (void)mouseUp:(NSEvent *)event
{
}

- (void)mouseDragged:(NSEvent *)event
{
}

- (void)mouseMoved:(NSEvent *)event
{
}


- (void)loadTextureList
{
	NSString *textureListPath = [[NSBundle mainBundle] pathForResource:@"Textures" ofType:@"plist"];
	NSArray *texturesList = [[NSArray alloc] initWithContentsOfFile:textureListPath];
	NSEnumerator *enumerator = [texturesList objectEnumerator];
	
	id textureName = nil;
	while((textureName = [enumerator nextObject]) != nil)
	{
		[[OPTextureManager sharedManager] textureWithName:textureName];
	}
}

- (void)loadPlayerTextures
{
	NSArray *playerTextures = [NSArray arrayWithContentsOfFile:[[NSBundle mainBundle] pathForResource:@"Player Textures" ofType:@"plist"]];
	
	NSEnumerator *enumerator = [playerTextures objectEnumerator];
	NSString *textureName = nil;
	while((textureName = [enumerator nextObject]) != nil) {
		[player addTextureID:[[OPTextureManager sharedManager] textureWithName:textureName] withName:textureName];
	}
	[player setTexture:[player textureWithName:[playerTextures objectAtIndex:0]]];
}

- (void)loadEnemyTextures
{
	NSDictionary *enemyTextures = [NSDictionary dictionaryWithContentsOfFile:[[NSBundle mainBundle] pathForResource:@"Enemy Textures" ofType:@"plist"]];
	
	NSEnumerator *enumerator = [enemyTextures keyEnumerator];
	NSString *eTextureKey = nil;
	while((eTextureKey = [enumerator nextObject]) != nil) {
		NSArray *eTextureList = [enemyTextures objectForKey:eTextureKey];
		NSEnumerator *subEnumerator = [eTextureList objectEnumerator];
		NSString *subName = nil;
		while((subName = [subEnumerator nextObject]) != nil) {
			NSString *fullTextureName = [NSString stringWithFormat:@"%@_%@", eTextureKey, subName];
			[[OPTextureManager sharedManager] textureWithName:fullTextureName];
		}
	}
}


- (void)setPlayer:(id)inPlayer
{
	player = inPlayer;
	[[currentMap currentRoom] setCurrent];	
}

- (id)player
{
	return player;
}

- (void)spawnEnemy:(NSNotification *)notification
{
	NSDictionary *enemyData = [notification userInfo];

	Class enemyClass = NSClassFromString([enemyData objectForKey:@"Class"]);
	int xloc, yloc;
	
	BOOL objectAtSpawnPoint = YES;

	do {
		xloc = ((rand() % 14) + 1) * 32;
		yloc = ((rand() % 9) + 1) * 32;
		id theObject = [[currentMap currentRoom] tileOnObjectLayerAtX:xloc andY:yloc];
		if(theObject == nil) {
			objectAtSpawnPoint = NO;
		}
	}while(objectAtSpawnPoint);
	
	id newEnemy = [[enemyClass alloc] initWithLocation:OPMakePoint(xloc,yloc) andSize:OPMakeSize(32,32)];

	NSMutableDictionary *theEnemyData = [[NSMutableDictionary alloc] init];
	NSMutableDictionary *enemyTextures = [[NSMutableDictionary alloc] init];

	[theEnemyData setObject:[enemyData objectForKey:@"Boss"] forKey:@"Boss"];
	[theEnemyData setObject:[enemyData objectForKey:@"Variation"] forKey:@"Variation"];

	NSEnumerator *enumerator = [[enemyData objectForKey:@"Textures"] objectEnumerator];
	NSString *textureName = nil;
	while((textureName = [enumerator nextObject]) != nil) {
		NSNumber *theTextureID = [NSNumber numberWithUnsignedInt:[[OPTextureManager sharedManager] textureWithName:textureName]];
		[enemyTextures setObject:theTextureID forKey:textureName];
	}
	[theEnemyData setObject:enemyTextures forKey:@"Textures"];

	[(OPEnemy *)newEnemy setEnemyData:theEnemyData];
	[[[currentMap currentRoom] enemies] addObject:newEnemy];
}

- (void)createItem:(NSNotification *)notification
{
	NSDictionary *itemInfo = [notification object];
	OPPoint location = NSPointFromString([itemInfo objectForKey:@"Location"]);
	location.x = location.x /32; location.y = location.y / 32;



	NSMutableDictionary *actionsDict = [[NSMutableDictionary alloc] init];
	NSArray *args;
	
	args = [NSArray arrayWithObjects:[itemInfo objectForKey:@"ItemType"], [NSNumber numberWithInt:location.x],
					 [NSNumber numberWithInt:location.y], [itemInfo objectForKey:@"Texture_Name"], nil];
	
	[actionsDict setObject:args forKey:@"args"];
	NSArray *actions = [NSArray arrayWithObject:actionsDict];

	NSMutableDictionary *eventData = [NSDictionary dictionaryWithObjectsAndKeys:[NSNumber numberWithBool:NO], @"Repeatable",
						[itemInfo objectForKey:@"Location"], @"Location",
						actions, @"Actions", nil];
	
	OPTrigger *theItem = [[OPTrigger alloc] initWithEventData:eventData];
	[self spawnItem:theItem];
}

@end


@implementation OPGameController (OPRoomEventActions)
- (void)movePlayer:(OPPoint)toPoint
{
	[player setLocationX:(toPoint.x * 32)];
	[player setLocationY:(toPoint.y * 32)];
}

- (void)loadMap:(NSString *)mapName
{
	currentMap = [gameMaps objectForKey:mapName];
}

- (void)setRoom:(OPPoint)theRoom
{
	[currentMap setCurrentRoomAtX:theRoom.x andY:theRoom.y];
}
- (void)spawnItem:(OPTrigger *)trigger
{
	NSArray *arguments = [[[trigger actions] objectAtIndex:0] objectForKey:@"args"];
	OPItem *anItem = [[OPItem alloc] initWithType:[arguments objectAtIndex:0]];

	[anItem setLocationX:[[arguments objectAtIndex:1] intValue] * 32];
	[anItem setLocationY:[[arguments objectAtIndex:2] intValue] * 32];
	GLuint textureID = [[OPTextureManager sharedManager] textureWithName:[arguments objectAtIndex:3]];
	[anItem setTexture:textureID];
	[[currentMap currentRoom] addItem:anItem];
}

- (void)removeTree:(OPTrigger *)trigger
{
	NSArray *arguments = [[[trigger actions] objectAtIndex:0] objectForKey:@"args"];
	OPPoint	point = NSPointFromString([arguments objectAtIndex:0]);

	[[[currentMap currentRoom] layer:OPlayerObjects] setTile:nil atX:point.x andY:point.y];
	OPPoint drawPoint = OPMakePoint(point.x * 32, point.y * 32);
	OPSprite *newSprite = [[OPSprite alloc] initWithLocation:drawPoint andSize:OPMakeSize(32,32)];
	[newSprite setTexture:[[OPTextureManager sharedManager] textureWithName:@"stairs_green"]];
	[newSprite autorelease];
	[[[currentMap currentRoom] layer:OPLayerBackground] setTile:newSprite atX:point.x andY:point.y];

	drawPoint.y += 32;
	newSprite  = [[OPSprite alloc] initWithLocation:drawPoint andSize:OPMakeSize(32,32)];
	[newSprite setTexture:[[OPTextureManager sharedManager] textureWithName:@"tree_green"]];
	[newSprite autorelease];
	[[[currentMap currentRoom] layer:OPlayerObjects] setTile:newSprite atX:point.x andY:point.y + 1];
}

- (void)hurtPlayer:(OPTrigger *)trigger
{
	[player hurt];
}

- (void)healPlayer:(OPTrigger *)trigger
{
	[player heal];
}
@end
